#Diet and Cancer Outcome Model
#Author: David D. Kim, Tufts Medical Center (DKim3@tuftsmedicalcenter.org)
#Contact: DKim3@tuftsmedicalcenter.org
#Main Purpose: To develop a microsimulation model to link dietary chanages with cancer outcomes 

#Version: 1.0 (July 7, 2017: First working model development completed)
#Version: 1.1 (August 8, 2017: Population Weighting + Age-adjustment for Cancer Costs)
#Version: 1.2 (August 23, 2017: Incremental Results Updated)
#Version: 1.3 (October 20, 2017: Changed linear dose-response relation (RR ~ Meat_intake) to exponential relation (ln(RR) ~ Meat_intake) 
#Version: 1.4 (November 17, 2017: Revised the cancer mortality based on survival probabilities - Not using projected mortality)
#Version: 1.5 (December 22, 2017): Fixed mortality calculation (exponential multiplicative, instead of linear additive) and seperated out cost components + added comments
#Version: 1.6 (February 15, 2018): Updated relative risk of colorectal and stomach cancer
#Version: 1.7 (April 13, 2018): Fixed the issue with EOL calculation to reflect changes in mortality calculation
#Version: 2.0 (May 3, 2018): Fixed the issue with lifetime cycle model 
#Version: 2.2 (Janaury 14, 2019): Corrected processed meat intake (use SE, instead of SD)
################################################################################################

library(matrixStats)
library(rriskDistributions)

#Creating Working Directory 
setwd("C:/Users/dkim3/Box Sync/Working Manuscripts/2018/CEA of Processed Meat Policies/Submitted Version/AJPM/AJPM -R1/Source Code & Input Data")

#Setting a initial seed to get consistent results in probalistic sampling
set.seed(12345)

#Function to estimate gamma/beta parameters based on Mean and SD
estGammaParams <- function(mu, var) {
  beta <- var / mu
  alpha <- mu / beta
  return(params = list(alpha = alpha, beta = beta))
}

estBetaParams <- function(mu, var) {
  alpha <- ((1 - mu) / var - 1 / mu) * mu ^ 2
  beta <- alpha * (1 / mu - 1)
  return(params = list(alpha = alpha, beta = beta))
}

##########################
######  DATA INPUT  ######
##########################

#Description of 32 Subgroups - Labelled 1-32 (4 age groups, 2 sexes, 4 R/E groups)
subgroup_description <- read.csv("Data Inputs/Subgroup Description.csv")

#########################
### Population Inputs ###
pop_dist <- read.csv("Data Inputs/pop_dist.csv")
baseline_consumption <- as.matrix(read.csv("Data Inputs/baseline_consumption.csv"))

##########################
### Policy Formulation ###

#Proposed effect size of policy interventions 
#(i.e., how much the proposed intervention would reduce consumption of the processed meat)
policy_effect_median <- 0.09 # 0.1 = 10% reduction in a specific year
policy_effect_lowerbound <- 0.05 
policy_effect_upperbound <- 0.15

######################################################
### Relationship between Diet and Cancer Incidence ###

#IMPORTANT ASSUMPTIONS: Induction period between changes in diets and changes in the risk of developing cancer
induction_period <- 5

#Every 50g/day of additional consumption is associated with increasing the risk of developing colorectal cancer by 16% (2017 CUP). 
mu_RR_CRC <- 1.16
CI_lo_RR_CRC <- 1.08
CI_hi_RR_CRC <- 1.26

log_CI_lo_RR_CRC <- log(CI_lo_RR_CRC)
log_CI_hi_RR_CRC <- log(CI_hi_RR_CRC)

log_RR_CRC_mean <- log(mu_RR_CRC)
log_RR_CRC_SE <- ((log_CI_hi_RR_CRC-log_CI_lo_RR_CRC)/2)/1.96

#Every 50g/day of additional consumption is associated with increasing the risk of developing stomach cancer by 18% (2016 CUP). 
mu_RR_SC <- 1.18
CI_lo_RR_SC <- 1.01
CI_hi_RR_SC <- 1.38

log_CI_lo_RR_SC <- log(CI_lo_RR_SC)
log_CI_hi_RR_SC <- log(CI_hi_RR_SC)

log_RR_SC_mean <- log(mu_RR_SC)
log_RR_SC_SE <- ((log_CI_hi_RR_SC-log_CI_lo_RR_SC)/2)/1.96

#########################
### Cancer Statistics ###

#Colorectal Cancer Incidence and Mortality at Baseline by 32 subgroups

#Based on 2013 USCS data - crude rate by 32 subgroups 
CRC_incidence <- read.csv("Data Inputs/CRC_incidence.csv") #Colorectal cancer
SC_incidence <- read.csv("Data Inputs/SC_incidence.csv") #Stomach cancer

#Based on the 2013 SEER 5-year relative survival data by 32 subgroups: later, these data converted to 1-year probability of death 
CRC_survival <- read.csv("Data Inputs/CRC_Survival_Prob (5-Year).csv")
SC_survival <- read.csv("Data Inputs/SC_Survival_Prob (5-Year).csv")

#Projection on Colorectal Cancer Incidence and Mortality for the next 100 years
#These projections are based on historical changes (measured in average annual percent change) in cancer incidence rates from 1999 to 2013
#Seperate STATA codes are available upon request 
CRC_proj_incidence_1999_2013 <- read.csv("Data Inputs/CRC_Incidence_Projection (1999-2013).csv")
SC_proj_incidence_1999_2013 <- read.csv("Data Inputs/SC_Incidence_Projection (1999-2013).csv")

CRC_proj_incidence_1999_2013_SE <- read.csv("Data Inputs/CRC_Incidence_Projection (1999-2013)_SE.csv")
SC_proj_incidence_1999_2013_SE <- read.csv("Data Inputs/SC_Incidence_Projection (1999-2013)_SE.csv")


##########################
### Baseline Mortality ###

#Baseline mortality among the general population (From the CDC lifetable by age, gender, and race groups)
#Measured in probability of death bewteen age t and t+1
baseline_mortality <- read.csv("Data Inputs/baseline_mortality.csv")

####################
### Cancer Costs ###

#Age adjusting factors - assuming that younger patients would receive more aggressive treatments 

#For the initial stage of cancer
age_factor_initial_lowerbound <- 1.0
age_factor_initial_median <- 1.2 #Multiply by 1.2 for age under 
age_factor_initial_upperbound <- 1.5

#For the end of life care
age_factor_EOL_lowerbound <- 1.0
age_factor_EOL_median <- 1.5 #Multiply by 1.5 for age under 65
age_factor_EOL_upperbound <- 2.0 

#Cancer-related heatlh care expenditures in three stages of cancer: 1) initial treatment stage, 2) continuous stage, and 3) end-of-life stage
HCE_CRC_initial_male <- 66153.136
HCE_CRC_initial_female <- 65534.952
HCE_CRC_continuous_male <- 4889.08
HCE_CRC_continuous_female <- 3361.176
HCE_CRC_EOL_male <- 91153.944
HCE_CRC_EOL_female <- 89928.216

HCE_SC_initial_male <- 83473.992
HCE_SC_initial_female <- 75624.864
HCE_SC_continuous_male <- 4556.048
HCE_SC_continuous_female <- 4231.528
HCE_SC_EOL_male <- 113986.32
HCE_SC_EOL_female <- 110398.512

#Time costs: indirect costs of cancer care because patients have to incur the loss of productive time  
patient_time_no_cancer_under65 <- 13.6 
patient_time_cancer_under65 <- 30.2  
patient_time_no_cancer_over65 <- 36.6 
patient_time_cancer_over65 <- 55.1 

#Time costs were valued based on hourly wage (Source: BLS)
wage_hourly <- 24.46 #in 2014 USD

#Productivity costs: long-term impact on the loss of productivity (e.g., lack of formal labor force participation due to cancer) 
#Assume log-normal distribution
Mean_LossProd_CRC <- 5935 #In 2014 USD
CI_lo_LossProd_CRC <- 2515
CI_hi_LossProd_CRC <- 11703

log_CI_lo_LossProd_CRC <- log(CI_lo_LossProd_CRC)
log_CI_hi_LossProd_CRC <- log(CI_hi_LossProd_CRC)

log_LossProd_CRC_mean <- log(Mean_LossProd_CRC)
log_LossProd_CRC_SE <- ((log_CI_hi_LossProd_CRC-log_CI_lo_LossProd_CRC)/2)/1.96

######################################################
### Health Care expenditures in General Population (background health care expenditure###
HCE_general <- read.csv("Data Inputs/HCE_general_pop.csv")
HCE_general_EOL_input <- 40956 #In 2014 USD

##############################################
### Health-related qualtiy of life (HRQoL) ###
QOL_CRC_initial_input <- cbind(0.76,0.03008)
QOL_CRC_continuous_input <- cbind(0.835,0.024063)
QOL_CRC_EOL_input <- cbind(0.643,0.04857)

QOL_SC_initial_input <- cbind(0.84,0.0357) #Based on the early stage
QOL_SC_continuous_input <- cbind(0.86,0.0346) #Based on curative treatment stage
QOL_SC_EOL_input <- cbind(0.68,0.0915) #Based on the Late Stage 

#########################################################
##############        SIMULATION          ###############
#########################################################
n.sim = 1000

#Creating empty arrays that results will be stored
CE.results_subgroup <- array(dim=c(2,11,32)) 
colnames(CE.results_subgroup) <- c("Discounted LYs","Discounted QALYs", "CRC Cases", "CRC PYs", "CRC Death", "SC Cases", "SC PYs", "SC Death", "Medical Costs", "Time Costs", "Productivity Loss")
rownames(CE.results_subgroup) <- c("No Policy","Policy")
dimnames(CE.results_subgroup)[[3]] <- subgroup_description[,2]

CE.results_subgroup_LowerBound <- array(dim=c(2,11,32)) 
CE.results_subgroup_UpperBound <- array(dim=c(2,11,32)) 

CE.incr.results_subgroup <- array(dim=c(3,11,32)) 
CE.incr.results_weighted_subgroup <- array(dim=c(3,11,32))

CE.results_weighted_subgroup <- array(dim=c(2,11,32))
CE.results_weighted_subgroup_LowerBound <- array(dim=c(2,11,32))
CE.results_weighted_subgroup_UpperBound <- array(dim=c(2,11,32))


CE.results_population = matrix(NA, nrow=2, ncol = 11) 
colnames(CE.results_population) <- c("Discounted LYs","Discounted QALYs", "CRC Cases", "CRC PYs", "CRC Death", "SC Cases", "SC PYs", "SC Death", "Medical Costs", "Time Costs", "Productivity Loss")
rownames(CE.results_population) <- c("No Policy","Policy")

CE.results_population_LowerBound = matrix(NA, nrow=2, ncol = 11) 
colnames(CE.results_population_LowerBound) <- c("Discounted LYs","Discounted QALYs", "CRC Cases", "CRC PYs", "CRC Death", "SC Cases", "SC PYs", "SC Death", "Medical Costs", "Time Costs", "Productivity Loss")
rownames(CE.results_population_LowerBound) <- c("No Policy","Policy")

CE.results_population_UpperBound = matrix(NA, nrow=2, ncol = 11) 
colnames(CE.results_population_UpperBound) <- c("Discounted LYs","Discounted QALYs", "CRC Cases", "CRC PYs", "CRC Death", "SC Cases", "SC PYs", "SC Death", "Medical Costs", "Time Costs", "Productivity Loss")
rownames(CE.results_population_UpperBound) <- c("No Policy","Policy")

CE.results_population = matrix(NA, nrow=2, ncol = 11) 
colnames(CE.results_population) <- c("Discounted LYs","Discounted QALYs", "CRC Cases", "CRC PYs", "CRC Death", "SC Cases", "SC PYs", "SC Death", "Medical Costs", "Time Costs", "Productivity Loss")
rownames(CE.results_population) <- c("No Policy","Policy")

CE.incr.results_population <- matrix(NA, nrow=3, ncol = 11)
colnames(CE.incr.results_population) <- c("Discounted LYs","Discounted QALYs", "CRC Cases", "CRC PYs", "CRC Death", "SC Cases", "SC PYs", "SC Death", "Medical Costs", "Time Costs", "Productivity Loss")
rownames(CE.incr.results_population) <- c("Lower Bound","Mean", "Upper Bound")


#Running the same CE simulation model for 32 subgroups
for (id in 1:32) {

id_subgroup <- id
print(id_subgroup)

###Simulating Baseline Levels of Processed Meat Intake

##Using two part approach because of a large proportion of no-meat eaters in the popualtion
#Simulated A Proportion of No Meat-Eaters Using Binomial Distribution
meat_eater <- as.matrix(rbinom(n.sim,1,baseline_consumption[id_subgroup,3]))
mean(meat_eater)

#Simulated Baseline Intake Among Meat-Eaters Based on Gamma Distribution 
intake_gamma_par <- estGammaParams(baseline_consumption[id_subgroup,4],baseline_consumption[id_subgroup,5]^2)
baseline_intake_meateaters <- as.matrix(rgamma(n.sim,intake_gamma_par$alpha,1/intake_gamma_par$beta))

baseline_intake <- meat_eater*baseline_intake_meateaters

#Descriptive statistics
sum(baseline_intake==0)
mean(baseline_intake)
mean(baseline_intake[baseline_intake>0])
sd(baseline_intake)
sd(baseline_intake[baseline_intake>0])
quantile(baseline_intake) 
quantile(baseline_intake[baseline_intake>0])

###Simulating Proposed level of intake after policy intervention

##Option 1: One-time effect 
#The 'rriskDistribution' package fits distributions to given data or known quantiles 
library(rriskDistributions)
policy_effect_beta_par <- get.beta.par(p=c(0.025,0.5,0.975),q=c(policy_effect_lowerbound, policy_effect_median, policy_effect_upperbound),fit.weights=c(1,1,1))

#Simulated policy effect given the lower and upper bound
policy_effect <- rbeta(n.sim,policy_effect_beta_par[1],policy_effect_beta_par[2])
summary(policy_effect)

#Simulated Proposed intake after policy intervention
proposed_intake <- baseline_intake*(1-policy_effect)
diff_intake = proposed_intake - baseline_intake

mean(diff_intake)
sd(diff_intake)
mean(diff_intake[baseline_intake > 0])
sd(baseline_intake > 0)

#Summarizing the Proposed Change in Year 0
intake_change <- cbind(baseline_intake,policy_effect,proposed_intake,diff_intake)
colnames(intake_change) <- c("baseline_intake","policy_effect_size","proposed_intake","difference_intake")
summary(intake_change)

###Simulating effect size

##Colorectal cancer
lognormal_CRC <- rnorm(n.sim,log_RR_CRC_mean,log_RR_CRC_SE)
RR_CRC <- exp(lognormal_CRC)

#Calibration to the literature estimates
quantile(RR_CRC, c(.025, .50, .975)) 

RR_CRC_baseline <- exp(log(RR_CRC)/50*baseline_intake)
RR_CRC_proposed <- exp(log(RR_CRC)/50*proposed_intake)
RR_CRC_difference <- exp(log(RR_CRC)/50*diff_intake)

quantile(RR_CRC_baseline, c(.025, .50, .975)) 
quantile(RR_CRC_proposed, c(.025, .50, .975)) 
quantile(RR_CRC_difference, c(.025, .50, .975))
quantile(RR_CRC_difference[RR_CRC_difference<1], c(.025, .50, .975)) 

#Stomach cancer
lognormal_SC <- rnorm(n.sim,log_RR_SC_mean,log_RR_SC_SE)
RR_SC <- exp(lognormal_SC)

quantile(RR_SC, c(.025, .50, .975)) 

RR_SC_baseline <- exp(log(RR_SC)/50*baseline_intake)
RR_SC_proposed <- exp(log(RR_SC)/50*proposed_intake)
RR_SC_difference <- exp(log(RR_SC)/50*diff_intake)

quantile(RR_SC_baseline, c(.025, .50, .975)) 
quantile(RR_SC_proposed, c(.025, .50, .975)) 
quantile(RR_SC_difference, c(.025, .50, .975)) 
quantile(RR_SC_difference[RR_SC_difference<1], c(.025, .50, .975)) 

#Colorectal cancer
p_CRC_incidence = 1 - exp(-(CRC_incidence[,2:3]/100000))
CRC_incidence_beta_par <- estBetaParams(p_CRC_incidence[id_subgroup,1],p_CRC_incidence[id_subgroup,2]^2)
p_CRC_proj_incidence_2013 <- as.matrix(rbeta(n.sim,CRC_incidence_beta_par$alpha,CRC_incidence_beta_par$beta))

CRC_survival_beta_par <- estBetaParams(CRC_survival[id_subgroup,3],CRC_survival[id_subgroup,4]^2)
p_CRC_survival_5yrs <- as.matrix(rbeta(n.sim,CRC_survival_beta_par$alpha,CRC_survival_beta_par$beta))

p_CRC_death_5yrs <- 1 - p_CRC_survival_5yrs
p_CRC_mortality <- 1 - exp(log(1-p_CRC_death_5yrs)/5)

p_CRC_proj_incidence_1999_2013 = cbind(CRC_proj_incidence_1999_2013[,2],(1 - exp(-(CRC_proj_incidence_1999_2013[,3:10]/100000))))
colnames(p_CRC_proj_incidence_1999_2013) = colnames(CRC_proj_incidence_1999_2013[2:10])
head(p_CRC_proj_incidence_1999_2013)


#Stomach Cancer
p_SC_incidence = 1 - exp(-(SC_incidence[,2:3]/100000))
SC_incidence_beta_par <- estBetaParams(p_SC_incidence[id_subgroup,1],p_SC_incidence[id_subgroup,2]^2)
p_SC_proj_incidence_2013 <- as.matrix(rbeta(n.sim,SC_incidence_beta_par$alpha,SC_incidence_beta_par$beta))

SC_survival_beta_par <- estBetaParams(SC_survival[id_subgroup,3],SC_survival[id_subgroup,4]^2)
p_SC_survival_5yrs <- as.matrix(rbeta(n.sim,SC_survival_beta_par$alpha,SC_survival_beta_par$beta))

p_SC_death_5yrs <- 1 - p_SC_survival_5yrs
p_SC_mortality <- 1 - exp(log(1-p_SC_death_5yrs)/5)

quantile(p_SC_mortality, c(.025, .50, .975)) 

p_SC_proj_incidence_1999_2013 = cbind(SC_proj_incidence_1999_2013[,2],(1 - exp(-(SC_proj_incidence_1999_2013[,3:10]/100000))))
colnames(p_SC_proj_incidence_1999_2013) = colnames(SC_proj_incidence_1999_2013[2:10])
head(p_SC_proj_incidence_1999_2013)

###COSTS###

### Age-adjustment factors for Cancer Costs ###
age_factor_initial_gam_par <- get.gamma.par(p=c(0.025,0.5,0.975),q=c(age_factor_initial_lowerbound, age_factor_initial_median, age_factor_initial_upperbound),fit.weights=c(1,1,1))
age_factor_initial <- rgamma(n.sim, age_factor_initial_gam_par[1], age_factor_initial_gam_par[2])
summary(age_factor_initial)

age_factor_EOL_gam_par <- get.gamma.par(p=c(0.025,0.5,0.975),q=c(age_factor_EOL_lowerbound, age_factor_EOL_median, age_factor_EOL_upperbound),fit.weights=c(1,1,1))
age_factor_EOL <- rgamma(n.sim, age_factor_EOL_gam_par[1], age_factor_EOL_gam_par[2])
summary(age_factor_EOL)

#Cancer Stage & gender specific HCE
#Colorectal Cancer
HCE_CRC_initial_male_gamma_par <- estGammaParams(HCE_CRC_initial_male,(HCE_CRC_initial_male*0.2)^2)
HCE_CRC_initial_male <- rgamma(n.sim,HCE_CRC_initial_male_gamma_par$alpha,1/HCE_CRC_initial_male_gamma_par$beta)

HCE_CRC_initial_female_gamma_par <- estGammaParams(HCE_CRC_initial_female,(HCE_CRC_initial_female*0.2)^2)
HCE_CRC_initial_female <- rgamma(n.sim,HCE_CRC_initial_female_gamma_par$alpha,1/HCE_CRC_initial_female_gamma_par$beta)

HCE_CRC_continuous_male_gamma_par <- estGammaParams(HCE_CRC_continuous_male,(HCE_CRC_continuous_male*0.2)^2)
HCE_CRC_continuous_male <- rgamma(n.sim,HCE_CRC_continuous_male_gamma_par$alpha,1/HCE_CRC_continuous_male_gamma_par$beta)

HCE_CRC_continuous_female_gamma_par <- estGammaParams(HCE_CRC_continuous_female,(HCE_CRC_continuous_female*0.2)^2)
HCE_CRC_continuous_female <- rgamma(n.sim,HCE_CRC_continuous_female_gamma_par$alpha,1/HCE_CRC_continuous_female_gamma_par$beta)

HCE_CRC_EOL_male_gamma_par <- estGammaParams(HCE_CRC_EOL_male,(HCE_CRC_EOL_male*0.2)^2)
HCE_CRC_EOL_male <- rgamma(n.sim,HCE_CRC_EOL_male_gamma_par$alpha,1/HCE_CRC_EOL_male_gamma_par$beta)

HCE_CRC_EOL_female_gamma_par <- estGammaParams(HCE_CRC_EOL_female,(HCE_CRC_EOL_female*0.2)^2)
HCE_CRC_EOL_female <- rgamma(n.sim,HCE_CRC_EOL_female_gamma_par$alpha,1/HCE_CRC_EOL_female_gamma_par$beta)

#Stomach Cancer
HCE_SC_initial_male_gamma_par <- estGammaParams(HCE_SC_initial_male,(HCE_SC_initial_male*0.2)^2)
HCE_SC_initial_male <- rgamma(n.sim,HCE_SC_initial_male_gamma_par$alpha,1/HCE_SC_initial_male_gamma_par$beta)

HCE_SC_initial_female_gamma_par <- estGammaParams(HCE_SC_initial_female,(HCE_SC_initial_female*0.2)^2)
HCE_SC_initial_female <- rgamma(n.sim,HCE_SC_initial_female_gamma_par$alpha,1/HCE_SC_initial_female_gamma_par$beta)

HCE_SC_continuous_male_gamma_par <- estGammaParams(HCE_SC_continuous_male,(HCE_SC_continuous_male*0.2)^2)
HCE_SC_continuous_male <- rgamma(n.sim,HCE_SC_continuous_male_gamma_par$alpha,1/HCE_SC_continuous_male_gamma_par$beta)

HCE_SC_continuous_female_gamma_par <- estGammaParams(HCE_SC_continuous_female,(HCE_SC_continuous_female*0.2)^2)
HCE_SC_continuous_female <- rgamma(n.sim,HCE_SC_continuous_female_gamma_par$alpha,1/HCE_SC_continuous_female_gamma_par$beta)

HCE_SC_EOL_male_gamma_par <- estGammaParams(HCE_SC_EOL_male,(HCE_SC_EOL_male*0.2)^2)
HCE_SC_EOL_male <- rgamma(n.sim,HCE_SC_EOL_male_gamma_par$alpha,1/HCE_SC_EOL_male_gamma_par$beta)

HCE_SC_EOL_female_gamma_par <- estGammaParams(HCE_SC_EOL_female,(HCE_SC_EOL_female*0.2)^2)
HCE_SC_EOL_female <- rgamma(n.sim,HCE_SC_EOL_female_gamma_par$alpha,1/HCE_SC_EOL_female_gamma_par$beta)

#Extracing gender-specific health care costs 
if(subgroup_description$Sex[id_subgroup] == "male") {
  HCE_CRC_initial <- HCE_CRC_initial_male
  HCE_CRC_continuous <- HCE_CRC_continuous_male
  HCE_CRC_EOL <- HCE_CRC_EOL_male 
  
  HCE_SC_initial <- HCE_SC_initial_male
  HCE_SC_continuous <- HCE_SC_continuous_male
  HCE_SC_EOL <- HCE_SC_EOL_male 
  
  
} else {
  HCE_CRC_initial <- HCE_CRC_initial_female
  HCE_CRC_continuous <- HCE_CRC_continuous_female
  HCE_CRC_EOL <- HCE_CRC_EOL_female 
  
  HCE_SC_initial <- HCE_SC_initial_female
  HCE_SC_continuous <- HCE_SC_continuous_female
  HCE_SC_EOL <- HCE_SC_EOL_female 
} 


#Time Costs: Age-specific time costs
p_timecost_no_cancer_under65 <- wage_hourly*rnorm(n.sim, patient_time_no_cancer_under65, patient_time_no_cancer_under65*0.2)
p_timecost_cancer_under65 <- wage_hourly*rnorm(n.sim, patient_time_cancer_under65, patient_time_cancer_under65*0.2)
incr_timecost_cancer_under65 <- p_timecost_cancer_under65 - p_timecost_no_cancer_under65

p_timecost_no_cancer_over65 <- wage_hourly*rnorm(n.sim, patient_time_no_cancer_over65, patient_time_no_cancer_over65*0.2)
p_timecost_cancer_over65 <- wage_hourly*rnorm(n.sim, patient_time_cancer_over65, patient_time_cancer_over65*0.2)
incr_timecost_cancer_over65 <- p_timecost_cancer_over65 - p_timecost_no_cancer_over65

#Productivity Costs
LossProd_CRC <- rnorm(n.sim,log_LossProd_CRC_mean,log_LossProd_CRC_SE)
LossProd_CRC <- exp(LossProd_CRC)
quantile(LossProd_CRC, c(.025, .50, .975)) 

# Health Care expenditures in General Population
HCE_general_pop_gamma_par <- estGammaParams(HCE_general[id_subgroup,2],HCE_general[id_subgroup,3]^2)
HCE_general_pop <- rgamma(n.sim,HCE_general_pop_gamma_par$alpha,1/HCE_general_pop_gamma_par$beta)


HCE_general_EOL_gamma_par <- estGammaParams(HCE_general_EOL_input,(HCE_general_EOL_input*0.2)^2)
HCE_general_EOL <- rgamma(n.sim, HCE_general_EOL_gamma_par$alpha,1/HCE_general_EOL_gamma_par$beta)


###HRQOL###
#Colorectal Cancer
QOL_CRC_initial_beta_par <- estBetaParams(QOL_CRC_initial_input[1],QOL_CRC_initial_input[2]^2)
QOL_CRC_initial <- rbeta(n.sim,QOL_CRC_initial_beta_par$alpha, QOL_CRC_initial_beta_par$beta)

QOL_CRC_continuous_beta_par <- estBetaParams(QOL_CRC_continuous_input[1],QOL_CRC_continuous_input[2]^2)
QOL_CRC_continuous <- rbeta(n.sim,QOL_CRC_continuous_beta_par$alpha, QOL_CRC_continuous_beta_par$beta)

QOL_CRC_EOL_beta_par <- estBetaParams(QOL_CRC_EOL_input[1],QOL_CRC_EOL_input[2]^2)
QOL_CRC_EOL <- rbeta(n.sim,QOL_CRC_EOL_beta_par$alpha, QOL_CRC_EOL_beta_par$beta)

#Stomach Cancer
QOL_SC_initial_beta_par <- estBetaParams(QOL_SC_initial_input[1],QOL_SC_initial_input[2]^2)
QOL_SC_initial <- rbeta(n.sim,QOL_SC_initial_beta_par$alpha, QOL_SC_initial_beta_par$beta)

QOL_SC_continuous_beta_par <- estBetaParams(QOL_SC_continuous_input[1],QOL_SC_continuous_input[2]^2)
QOL_SC_continuous <- rbeta(n.sim,QOL_SC_continuous_beta_par$alpha, QOL_SC_continuous_beta_par$beta)

QOL_SC_EOL_beta_par <- estBetaParams(QOL_SC_EOL_input[1],QOL_SC_EOL_input[2]^2)
QOL_SC_EOL <- rbeta(n.sim,QOL_SC_EOL_beta_par$alpha, QOL_SC_EOL_beta_par$beta)

###################################################
##############    LIFETIME MODEL    ###############
###################################################

#Creating 8 different health states 
n.states <- 8
states <- c("Healthy/No Cancer", "CRC_Initial", "CRC_Continous", "CRC_Death", "SC_Initial", "SC_Continous", "SC_Death", "Non-Cancer Death")

## of interventions: No policy vs. Policy
n.intervention <- 2

#Number of Lifetime Cycles
n.cycles = 101 - subgroup_description[id_subgroup,3]

#discounting rate
beta = 0.03

###Initializing Matrices  
#Storing Results
CRC_case_No_Policy <- 0
CRC_case_Policy <- 0
Incr_CRC_case <- 0

SC_case_No_Policy<- 0
SC_case_Policy <- 0
Incr_SC_case <- 0

CRC_PY_No_Policy <- 0
CRC_PY_Policy <- 0
Incr_CRC_PY <- 0

SC_PY_No_Policy<- 0
SC_PY_Policy <- 0
Incr_SC_PY <- 0

CRC_death_No_Policy <- 0
CRC_death_Policy <- 0
Incr_CRC_death <- 0

SC_death_No_Policy<- 0
SC_death_Policy <- 0
Incr_SC_death <- 0

LY_No_Policy <- 0
LY_Policy <- 0
Incr_LY <- 0

LY_disc_No_Policy <- 0
LY_disc_Policy <- 0
Incr_LY_disc <- 0

QALY_No_Policy <- 0
QALY_Policy <- 0
Incr_QALY <- 0

QALY_disc_No_Policy <- 0
QALY_disc_Policy <- 0
Incr_QALY_disc <- 0

Cost_No_Policy <- 0
Cost_Policy <- 0
Incr_Cost <- 0 

Time_Cost_No_Policy <- 0
Time_Cost_Policy <- 0
Incr_Time_Cost <- 0

Prod_Cost_No_Policy <- 0
Prod_Cost_Policy <- 0
Incr_Prod_Cost <- 0

#Treatment Effects (when reduction in processed meat consumption would influence reduction in RR of developing cancer) based on assumptions: induction periods & policy impact
RR_CRC_adjustment <- matrix(NA, nrow=n.cycles, ncol = n.sim)
RR_SC_adjustment <- matrix(NA, nrow=n.cycles, ncol = n.sim)

Healthy.2.CRC_initial_adjusted <- matrix(NA, nrow=n.cycles, ncol = n.sim)
Healthy.2.SC_initial_adjusted <- matrix(NA, nrow=n.cycles, ncol = n.sim)
Healthy.2.Healthy_adjusted <- matrix(NA, nrow=n.cycles, ncol = n.sim)

for(t in 1:n.cycles) {
  if(t < induction_period) {
    RR_CRC_adjustment[t,] <- 1
    RR_SC_adjustment[t,] <- 1
  } else {
    RR_CRC_adjustment[t,] <- RR_CRC_difference
    RR_SC_adjustment[t,] <- RR_SC_difference
  } 
}


#Transition Matrices for the No Intervention and Intervention Groups
array_No_Policy <- array(dim=c(n.states,n.states,n.cycles)) 
array_Policy  <- array(dim=c(n.states,n.states,n.cycles)) 

rownames(array_No_Policy) <- states
colnames(array_No_Policy) <- states
dimnames(array_No_Policy)[[3]] <- paste("Year", c(1:n.cycles))

rownames(array_Policy) <- states
colnames(array_Policy) <- states
dimnames(array_Policy)[[3]] <- paste("Year", c(1:n.cycles))

array_CRC_No_Policy <- array(dim=c(n.states,n.states,n.cycles)) 
array_CRC_Policy  <- array(dim=c(n.states,n.states,n.cycles)) 

array_SC_No_Policy <- array(dim=c(n.states,n.states,n.cycles)) 
array_SC_Policy  <- array(dim=c(n.states,n.states,n.cycles)) 

rownames(array_CRC_No_Policy) <- states
colnames(array_CRC_No_Policy) <- states
dimnames(array_CRC_No_Policy)[[3]] <- paste("Year", c(1:n.cycles))

rownames(array_SC_No_Policy) <- states
colnames(array_SC_No_Policy) <- states
dimnames(array_SC_No_Policy)[[3]] <- paste("Year", c(1:n.cycles))

rownames(array_CRC_Policy) <- states
colnames(array_CRC_Policy) <- states
dimnames(array_CRC_Policy)[[3]] <- paste("Year", c(1:n.cycles))

rownames(array_SC_Policy) <- states
colnames(array_SC_Policy) <- states
dimnames(array_SC_Policy)[[3]] <- paste("Year", c(1:n.cycles))

#Annual Costs and HRQoL Specific to Health States
input_cost <- matrix(NA, nrow=n.intervention, ncol = n.states)
input_hrqol <- matrix(NA, nrow=n.intervention, ncol = n.states)
  
#Consequences
lifeyears <-array(dim=c(n.intervention,n.states,n.cycles+1))
qalys <-array(dim=c(n.intervention,n.states,n.cycles+1))
LY_disc <-array(dim=c(n.intervention,n.states,n.cycles+1))
QALY_disc <-array(dim=c(n.intervention,n.states,n.cycles+1))
costs <-array(dim=c(n.intervention,n.states,n.cycles+1)) 
time_costs <- matrix(NA, nrow=n.intervention, ncol=n.cycles)
prod_costs <- matrix(NA, nrow=n.intervention, ncol=n.cycles)
  
CRC <-array(dim=c(n.intervention,n.states,n.cycles+1))
SC <-array(dim=c(n.intervention,n.states,n.cycles+1))

CRC_attributable_death_ratio <- matrix(NA, nrow=n.cycles)
SC_attributable_death_ratio <- matrix(NA, nrow=n.cycles)

#Set Initial Conditions
start_pop <- matrix(c(1,0,0,0,0,0,0,0,
                      1,0,0,0,0,0,0,0),
                    nrow = n.intervention, ncol = n.states, byrow=T)

#For each of simulated values
for(s in 1:n.sim) {
  print(s)
  #Loop through cycles
  for(t in 1: n.cycles) {
    #Creating Transition probabilites: 5 x 5 matrics 
    age_cycle <- subgroup_description[id_subgroup,3] + (t-1)
    
      #Healthy to other States
        Healthy.2.CRC_initial <- p_CRC_proj_incidence_1999_2013[t,as.character(subgroup_description$Sex_Race[id_subgroup])]
        Healthy.2.CRC_continuous <- 0
        Healthy.2.CRC_death <- 0
        
        Healthy.2.SC_initial <- p_SC_proj_incidence_1999_2013[t,as.character(subgroup_description$Sex_Race[id_subgroup])]
        Healthy.2.SC_continuous <- 0
        Healthy.2.SC_death <- 0
        
        Healthy.2.Non_cancer_death <- baseline_mortality[age_cycle,as.character(subgroup_description$Sex_Race[id_subgroup])]
        
        Healthy.2.Healthy <- 1 - sum(Healthy.2.CRC_initial, Healthy.2.CRC_continuous, Healthy.2.CRC_death, Healthy.2.SC_initial, Healthy.2.SC_continuous, Healthy.2.SC_death, Healthy.2.Non_cancer_death)
      
      #CRC_Initial to other states
        CRC_initial.2.Healthy <- 0
        
        CRC_initial.2.CRC_initial <- 0
        CRC_initial.2.CRC_death <- 0 #In the revised model, this sets to zero and incorporated into CRC to death combined to correct potential over-estimates (change from linear additive model to expoential death - see code below)
        
        CRC_initial.2.SC_initial <- 0
        CRC_initial.2.SC_continuous <- 0 
        CRC_initial.2.SC_death <- 0
        
        CRC.2.death_combined <- -log(1-baseline_mortality[age_cycle,as.character(subgroup_description$Sex_Race[id_subgroup])]) + -log(1-p_CRC_mortality[s])
        CRC_attributable_death_ratio[t,1] <- -log(1-p_CRC_mortality[s]) / CRC.2.death_combined
        CRC_initial.2.overall_death <-  1 - exp(-CRC.2.death_combined)
        
        CRC_initial.2.CRC_continuous <- 1 - sum(CRC_initial.2.Healthy, CRC_initial.2.CRC_initial, CRC_initial.2.CRC_death, CRC_initial.2.SC_initial, CRC_initial.2.SC_continuous, CRC_initial.2.SC_death, CRC_initial.2.overall_death)
      
      #SC_Initial to other states
        SC_initial.2.Healthy <- 0
        
        SC_initial.2.CRC_initial <- 0
        SC_initial.2.CRC_continuous <- 0 
        SC_initial.2.CRC_death <- 0 
        
        SC_initial.2.SC_initial <- 0
        SC_initial.2.SC_death <- 0 #See the note under "CRC_initial.2.CRC_death"
        
        SC.2.death_combined <- -log(1-baseline_mortality[age_cycle,as.character(subgroup_description$Sex_Race[id_subgroup])]) + -log(1-p_SC_mortality[s])
        SC_attributable_death_ratio[t,1] <- -log(1-p_SC_mortality[s]) / SC.2.death_combined
        SC_initial.2.overall_death <- 1 - exp(-SC.2.death_combined)
        
        SC_initial.2.SC_continuous <- 1 - sum(SC_initial.2.Healthy, SC_initial.2.CRC_initial, SC_initial.2.CRC_continuous, SC_initial.2.CRC_death, SC_initial.2.SC_initial, SC_initial.2.SC_death, SC_initial.2.overall_death)
        
      #Cancer_Continuous to other states
        CRC_continuous.2.Healthy <- 0
        
        CRC_continuous.2.CRC_initial <- 0
        CRC_continuous.2.CRC_death <- 0 #See the note under "CRC_initial.2.CRC_death"
        
        CRC_continuous.2.SC_initial <- 0
        CRC_continuous.2.SC_continuous <- 0
        CRC_continuous.2.SC_death <- 0
        
        CRC_continuous.2.overall_death <- 1 - exp(-CRC.2.death_combined)
        
        CRC_continuous.2.CRC_continuous <- 1 - sum(CRC_continuous.2.Healthy, CRC_continuous.2.CRC_initial, CRC_continuous.2.CRC_death, CRC_continuous.2.SC_initial, CRC_continuous.2.SC_continuous, CRC_continuous.2.SC_death, CRC_continuous.2.overall_death)
        
        
      #Cancer_Continuous to other states
        SC_continuous.2.Healthy <- 0
        
        SC_continuous.2.CRC_initial <- 0
        SC_continuous.2.CRC_continuous <- 0
        SC_continuous.2.CRC_death <- 0
        
        SC_continuous.2.SC_initial <- 0
        SC_continuous.2.SC_death <- 0 #See the note under "CRC_initial.2.CRC_death"
        
        SC_continuous.2.overall_death <- 1 - exp(-SC.2.death_combined)
        
        SC_continuous.2.SC_continuous <- 1 - sum(SC_continuous.2.Healthy, SC_continuous.2.CRC_initial, SC_continuous.2.CRC_continuous, SC_continuous.2.CRC_death, SC_continuous.2.SC_initial, SC_continuous.2.SC_death, SC_continuous.2.overall_death)
        
      #Death (Non-cancer death and Cancer_death) is an absorbing state (i.e., no other transition possibility) 
        Cancer_death.2.Cancer_death <- 1
        Non_cancer_death.2.Non_cancer_death <- 1
    
    #Transition probabilites for No Intervention Group    
    array_No_Policy[,,t] <- matrix(c(Healthy.2.Healthy, Healthy.2.CRC_initial, Healthy.2.CRC_continuous, Healthy.2.CRC_death, Healthy.2.SC_initial, Healthy.2.SC_continuous, Healthy.2.SC_death, Healthy.2.Non_cancer_death, 
                                     CRC_initial.2.Healthy, CRC_initial.2.CRC_initial, CRC_initial.2.CRC_continuous, CRC_initial.2.CRC_death, CRC_initial.2.SC_initial, CRC_initial.2.SC_continuous, CRC_initial.2.SC_death, CRC_initial.2.overall_death,
                                     CRC_continuous.2.Healthy, CRC_continuous.2.CRC_initial, CRC_continuous.2.CRC_continuous, CRC_continuous.2.CRC_death, CRC_continuous.2.SC_initial, CRC_continuous.2.SC_continuous, CRC_continuous.2.SC_death, CRC_continuous.2.overall_death, 
                                     0, 0, 0, Cancer_death.2.Cancer_death, 0, 0, 0, 0, 
                                     SC_initial.2.Healthy, SC_initial.2.CRC_initial, SC_initial.2.CRC_continuous, SC_initial.2.CRC_death, SC_initial.2.SC_initial, SC_initial.2.SC_continuous, SC_initial.2.SC_death, SC_initial.2.overall_death,
                                     SC_continuous.2.Healthy, SC_continuous.2.CRC_initial, SC_continuous.2.CRC_continuous, SC_continuous.2.CRC_death, SC_continuous.2.SC_initial, SC_continuous.2.SC_continuous, SC_continuous.2.SC_death, SC_continuous.2.overall_death, 
                                     0, 0, 0, 0, 0, 0, Cancer_death.2.Cancer_death, 0, 
                                     0, 0, 0, 0, 0, 0, 0, Non_cancer_death.2.Non_cancer_death), 
                                   nrow = n.states, ncol = n.states, byrow = TRUE)

    
    #Transition probabilities for Intervention Group
      #Adjusting Cancer Incidence
        #Healthy to other States
        Healthy.2.CRC_initial_adjusted[t,s] <- RR_CRC_adjustment[t,s]*p_CRC_proj_incidence_1999_2013[t+1,as.character(subgroup_description$Sex_Race[id_subgroup])]
        Healthy.2.SC_initial_adjusted[t,s] <- RR_SC_adjustment[t,s]*p_SC_proj_incidence_1999_2013[t+1,as.character(subgroup_description$Sex_Race[id_subgroup])]
        Healthy.2.Healthy_adjusted[t,s] <- 1 - sum(Healthy.2.CRC_initial_adjusted[t,s], Healthy.2.SC_initial_adjusted[t,s], Healthy.2.CRC_continuous, Healthy.2.CRC_continuous, Healthy.2.CRC_death, Healthy.2.SC_death, Healthy.2.Non_cancer_death)

    array_Policy[,,t] <- matrix(c(Healthy.2.Healthy_adjusted[t,s], Healthy.2.CRC_initial_adjusted[t,s], Healthy.2.CRC_continuous, Healthy.2.CRC_death, Healthy.2.SC_initial_adjusted[t,s], Healthy.2.SC_continuous, Healthy.2.SC_death, Healthy.2.Non_cancer_death, 
                                  CRC_initial.2.Healthy, CRC_initial.2.CRC_initial, CRC_initial.2.CRC_continuous, CRC_initial.2.CRC_death, CRC_initial.2.SC_initial, CRC_initial.2.SC_continuous, CRC_initial.2.SC_death, CRC_initial.2.overall_death,
                                  CRC_continuous.2.Healthy, CRC_continuous.2.CRC_initial, CRC_continuous.2.CRC_continuous, CRC_continuous.2.CRC_death, CRC_continuous.2.SC_initial, CRC_continuous.2.SC_continuous, CRC_continuous.2.SC_death, CRC_continuous.2.overall_death, 
                                  0, 0, 0, Cancer_death.2.Cancer_death, 0, 0, 0, 0, 
                                  SC_initial.2.Healthy, SC_initial.2.CRC_initial, SC_initial.2.CRC_continuous, SC_initial.2.CRC_death, SC_initial.2.SC_initial, SC_initial.2.SC_continuous, SC_initial.2.SC_death, SC_initial.2.overall_death,
                                  SC_continuous.2.Healthy, SC_continuous.2.CRC_initial, SC_continuous.2.CRC_continuous, SC_continuous.2.CRC_death, SC_continuous.2.SC_initial, SC_continuous.2.SC_continuous, SC_continuous.2.SC_death, SC_continuous.2.overall_death, 
                                  0, 0, 0, 0, 0, 0, Cancer_death.2.Cancer_death, 0, 
                                  0, 0, 0, 0, 0, 0, 0, Non_cancer_death.2.Non_cancer_death), 
                                nrow = n.states, ncol = n.states, byrow = TRUE)
  }

    #Incorporating Age Adjustment Factors
    if(age_cycle < 65) {
      input_cost[,] <- matrix(c(HCE_general_pop[s], HCE_CRC_initial[s]*age_factor_initial[s], HCE_CRC_continuous[s], 0, HCE_SC_initial[s]*age_factor_initial[s], HCE_SC_continuous[s], 0, 0,
                                HCE_general_pop[s], HCE_CRC_initial[s]*age_factor_initial[s], HCE_CRC_continuous[s], 0, HCE_SC_initial[s]*age_factor_initial[s], HCE_SC_continuous[s], 0, 0), 
                              nrow = n.intervention, ncol = n.states, byrow = TRUE)
    } else {
      input_cost[,] <- matrix(c(HCE_general_pop[s], HCE_CRC_initial[s], HCE_CRC_continuous[s], 0, HCE_SC_initial[s], HCE_SC_continuous[s], 0, 0,
                                HCE_general_pop[s], HCE_CRC_initial[s], HCE_CRC_continuous[s], 0, HCE_SC_initial[s], HCE_SC_continuous[s], 0, 0), 
                              nrow = n.intervention, ncol = n.states, byrow = TRUE)
    }
    
    input_hrqol[,] <- matrix(c(1, QOL_CRC_initial[s], QOL_CRC_continuous[s], 0, QOL_SC_initial[s], QOL_SC_continuous[s], 0, 0,
                               1, QOL_CRC_initial[s], QOL_CRC_continuous[s], 0, QOL_SC_initial[s], QOL_SC_continuous[s], 0, 0), 
                             nrow = n.intervention, ncol = n.states, byrow = TRUE)
    
    #Set Initial conditions:
    lifeyears[,,1] <- start_pop
    qalys[,,1] <- lifeyears[,,1]*input_hrqol[,]
    costs[,,1] <- lifeyears[,,1]*input_cost[,]
    LY_disc[,,1] <- lifeyears[,,1]
    QALY_disc[,,1] <- qalys[,,1]
    
    #Lifetime Transition
    # use loop to run through cycles (fancier way to do this?)
    max <- n.cycles+1 
      for (t in 2:max){
        
        # calculate life years (use matrix multiplication [%*%] to run cohort through transition matrices)
        lifeyears[1,,t] <- lifeyears[1,,t-1]%*%array_No_Policy[,,t-1]
        lifeyears[2,,t] <- lifeyears[2,,t-1]%*%array_Policy[,,t-1]
        
        #From the overall death occured in a specific cycle, re-estimate cancer-attributable death and put back into CRC_death and SC_death
        #Under no policy
        lifeyears[1,4,t] <- lifeyears[1,2:3,t-1]%*%array_No_Policy[2:3,8,t-1]*CRC_attributable_death_ratio[t-1,1]
        lifeyears[1,7,t] <- lifeyears[1,5:6,t-1]%*%array_No_Policy[5:6,8,t-1]*SC_attributable_death_ratio[t-1,1]
        
        #Under nutrition policy
        lifeyears[2,4,t] <- lifeyears[2,2:3,t-1]%*%array_Policy[2:3,8,t-1]*CRC_attributable_death_ratio[t-1,1]
        lifeyears[2,7,t] <- lifeyears[2,5:6,t-1]%*%array_Policy[5:6,8,t-1]*SC_attributable_death_ratio[t-1,1]
        
        # now multiply in utility matrix, discount, and half cycle correction:
        qalys[1,,t] <- lifeyears[1,,t]*input_hrqol[1,]
        qalys[2,,t] <- lifeyears[2,,t]*input_hrqol[2,]
        
        qalys[1,4,t] <- 0.5*qalys[1,,t-1]%*%array_No_Policy[,4,t-1]
        qalys[2,4,t] <- 0.5*qalys[2,,t-1]%*%array_Policy[,4,t-1]
        
        qalys[1,7,t] <- 0.5*qalys[1,,t-1]%*%array_No_Policy[,7,t-1]
        qalys[2,7,t] <- 0.5*qalys[2,,t-1]%*%array_Policy[,7,t-1]
        
        qalys[1,8,t] <- 0.5*qalys[1,,t-1]%*%array_No_Policy[,8,t-1]
        qalys[2,8,t] <- 0.5*qalys[2,,t-1]%*%array_Policy[,8,t-1]
        
        
        LY_disc[,,t] <- lifeyears[,,t]/(1+beta)^(t-1)
        QALY_disc[,,t] <- qalys[,,t]/(1+beta)^(t-1)
        
        # Age-dependent costs: health care costs (including EOL) + time costs + the loss of productivity
        if(age_cycle < 65) {
          costs[,,t]<-(lifeyears[,,t]*input_cost)/(1+beta)^(t-1)
          
          costs[1,4,t] <- (lifeyears[1,,t-1]%*%array_No_Policy[,4,t-1])*(HCE_CRC_EOL[s]*age_factor_EOL[s])/(1+beta)^(t-1)
          costs[2,4,t] <- (lifeyears[2,,t-1]%*%array_Policy[,4,t-1])*(HCE_CRC_EOL[s]*age_factor_EOL[s])/(1+beta)^(t-1)
          
          costs[1,7,t] <- (lifeyears[1,,t-1]%*%array_No_Policy[,7,t-1])*(HCE_SC_EOL[s]*age_factor_EOL[s])/(1+beta)^(t-1)
          costs[2,7,t] <- (lifeyears[2,,t-1]%*%array_Policy[,7,t-1])*(HCE_SC_EOL[s]*age_factor_EOL[s])/(1+beta)^(t-1)
          
          costs[1,8,t] <- (lifeyears[1,,t-1]%*%array_No_Policy[,8,t-1])*HCE_general_EOL[s]/(1+beta)^(t-1)
          costs[2,8,t] <- (lifeyears[2,,t-1]%*%array_Policy[,8,t-1])*HCE_general_EOL[s]/(1+beta)^(t-1)
          
          time_costs[1,t-1] <- (sum(lifeyears[1,c(2:3,5:6),t-1])*incr_timecost_cancer_under65[s])/(1+beta)^(t-1)
          time_costs[2,t-1] <- (sum(lifeyears[2,c(2:3,5:6),t-1])*incr_timecost_cancer_under65[s])/(1+beta)^(t-1)
          
          prod_costs[1,t-1] <- (sum(lifeyears[1,c(2:3,5:6),t-1])*LossProd_CRC[s])/(1+beta)^(t-1)
          prod_costs[2,t-1] <- (sum(lifeyears[2,c(2:3,5:6),t-1])*LossProd_CRC[s])/(1+beta)^(t-1)
          
        } else {
          costs[,,t]<-(lifeyears[,,t]*input_cost)/(1+beta)^(t-1)
          
          costs[1,4,t] <- (lifeyears[1,,t-1]%*%array_No_Policy[,4,t-1])*HCE_CRC_EOL[s]/(1+beta)^(t-1)
          costs[2,4,t] <- (lifeyears[2,,t-1]%*%array_Policy[,4,t-1])*HCE_CRC_EOL[s]/(1+beta)^(t-1)
          
          costs[1,7,t] <- (lifeyears[1,,t-1]%*%array_No_Policy[,7,t-1])*HCE_SC_EOL[s]/(1+beta)^(t-1)
          costs[2,7,t] <- (lifeyears[2,,t-1]%*%array_Policy[,7,t-1])*HCE_SC_EOL[s]/(1+beta)^(t-1)
          
          costs[1,8,t] <- (lifeyears[1,8,t]-(lifeyears[1,4,t]+lifeyears[1,7,t]))*HCE_general_EOL[s]/(1+beta)^(t-1)
          costs[2,8,t] <- (lifeyears[2,8,t]-(lifeyears[2,4,t]+lifeyears[2,7,t]))*HCE_general_EOL[s]/(1+beta)^(t-1)
          
          time_costs[1,t-1] <- (sum(lifeyears[1,c(2:3,5:6),t-1])*incr_timecost_cancer_over65[s])/(1+beta)^(t-1)
          time_costs[2,t-1] <- (sum(lifeyears[2,c(2:3,5:6),t-1])*incr_timecost_cancer_over65[s])/(1+beta)^(t-1)
          
          prod_costs[1,t-1] <- (sum(lifeyears[1,c(2:3,5:6),t-1])*LossProd_CRC[s])/(1+beta)^(t-1)
          prod_costs[2,t-1] <- (sum(lifeyears[2,c(2:3,5:6),t-1])*LossProd_CRC[s])/(1+beta)^(t-1)
        }
      }
    
    ###Lifetime Consequences 
    
    #Health Outcomes: LYs and QALYs
    LY_No_Policy[s] <- sum(lifeyears[1,c(1:3,5:6),]) + 0.5*sum(lifeyears[1,c(4,7,8),])
    LY_Policy[s] <- sum(lifeyears[2,c(1:3,5:6),]) + 0.5*sum(lifeyears[2,c(4,7,8),])
    Incr_LY[s] <- LY_Policy[s] - LY_No_Policy[s]
    
    LY_disc_No_Policy[s] <- sum(LY_disc[1,c(1:3,5:6),]) + 0.5*sum(LY_disc[1,c(4,7,8),])
    LY_disc_Policy[s] <- sum(LY_disc[2,c(1:3,5:6),]) + 0.5*sum(LY_disc[2,c(4,7,8),])
    Incr_LY_disc[s] <- LY_disc_Policy[s] - LY_disc_No_Policy[s]
    
    QALY_No_Policy[s] <- sum(qalys[1,,])
    QALY_Policy[s] <- sum(qalys[2,,])
    Incr_QALY[s] <- QALY_Policy[s] - QALY_No_Policy[s]
    
    QALY_disc_No_Policy[s] <- sum(QALY_disc[1,,])
    QALY_disc_Policy[s] <- sum(QALY_disc[2,,])
    Incr_QALY_disc[s] <- QALY_disc_Policy[s] - QALY_disc_No_Policy[s]
    
    
    #Cancer Outcomes: New Cases, Person-Years, Death
    CRC_case_No_Policy[s] <- sum(lifeyears[1,2,])
    CRC_case_Policy[s] <- sum(lifeyears[2,2,])
    Incr_CRC_case[s] <- CRC_case_Policy[s] - CRC_case_No_Policy[s]
    
    CRC_PY_No_Policy[s] <- sum(lifeyears[1,2:3,])
    CRC_PY_Policy[s] <- sum(lifeyears[2,2:3,])
    Incr_CRC_PY[s] <- CRC_PY_Policy[s] - CRC_PY_No_Policy[s]
    
    CRC_death_No_Policy[s] <- sum(lifeyears[1,4,])
    CRC_death_Policy[s] <- sum(lifeyears[2,4,])
    Incr_CRC_death[s] <- CRC_death_Policy[s] - CRC_death_No_Policy[s]
    
    SC_case_No_Policy[s] <- sum(lifeyears[1,5,])
    SC_case_Policy[s] <- sum(lifeyears[2,5,])
    Incr_SC_case[s] <- SC_case_Policy[s] - SC_case_No_Policy[s]
    
    SC_PY_No_Policy[s] <- sum(lifeyears[1,5:6,])
    SC_PY_Policy[s] <- sum(lifeyears[2,5:6,])
    Incr_SC_PY[s] <- SC_PY_Policy[s] - SC_PY_No_Policy[s]
    
    SC_death_No_Policy[s] <- sum(lifeyears[1,7,])
    SC_death_Policy[s] <- sum(lifeyears[2,7,])
    Incr_SC_death[s] <- SC_death_Policy[s] - SC_death_No_Policy[s]
    
    #Costs
    
    Cost_No_Policy[s] <- sum(costs[1,,])
    Cost_Policy[s] <- sum(costs[2,,])
    Incr_Cost[s] <- Cost_Policy[s] - Cost_No_Policy[s]
    
    Time_Cost_No_Policy[s] <- sum(time_costs[1,], na.rm=TRUE)
    Time_Cost_Policy[s] <- sum(time_costs[2,], na.rm=TRUE)
    Incr_Time_Cost[s] <- Time_Cost_Policy[s] - Time_Cost_No_Policy[s]
    
    Prod_Cost_No_Policy[s] <- sum(prod_costs[1,], na.rm=TRUE)
    Prod_Cost_Policy[s] <- sum(prod_costs[2,], na.rm=TRUE)
    Incr_Prod_Cost[s] <- Prod_Cost_Policy[s] - Prod_Cost_No_Policy[s] 
    
}

CE.results_subgroup[,,id_subgroup] <- matrix(c(mean(LY_disc_No_Policy), mean(QALY_disc_No_Policy), 
                                               mean(CRC_case_No_Policy), mean(CRC_PY_No_Policy), mean(CRC_death_No_Policy), 
                                               mean(SC_case_No_Policy), mean(SC_PY_No_Policy), mean(SC_death_No_Policy), 
                                               round(mean(Cost_No_Policy)), round(mean(Time_Cost_No_Policy)), round(mean(Prod_Cost_No_Policy)),
                                               mean(LY_disc_Policy), mean(QALY_disc_Policy), 
                                               mean(CRC_case_Policy), mean(CRC_PY_Policy), mean(CRC_death_Policy), 
                                               mean(SC_case_Policy), mean(SC_PY_Policy), mean(SC_death_Policy), 
                                               round(mean(Cost_Policy)), round(mean(Time_Cost_Policy)), round(mean(Prod_Cost_Policy))
                                               ), nrow=2, ncol = 11, byrow=T)

CE.results_subgroup_LowerBound[,,id_subgroup] <- matrix(c(quantile(LY_disc_No_Policy, c(.025)), quantile(QALY_disc_No_Policy, c(.025)), 
                                                          quantile(CRC_case_No_Policy, c(.025)), quantile(CRC_PY_No_Policy, c(.025)), quantile(CRC_death_No_Policy, c(.025)), 
                                                          quantile(SC_case_No_Policy, c(.025)), quantile(SC_PY_No_Policy, c(.025)), quantile(SC_death_No_Policy, c(.025)), 
                                                          round(quantile(Cost_No_Policy, c(.025))), round(quantile(Time_Cost_No_Policy, c(.025))), round(quantile(Prod_Cost_No_Policy, c(.025))), 
                                                          quantile(LY_disc_Policy, c(.025)), quantile(QALY_disc_Policy, c(.025)), 
                                                          quantile(CRC_case_Policy, c(.025)), quantile(CRC_PY_Policy, c(.025)), quantile(CRC_death_Policy, c(.025)), 
                                                          quantile(SC_case_Policy, c(.025)), quantile(SC_PY_Policy, c(.025)), quantile(SC_death_Policy, c(.025)), 
                                                          round(quantile(Cost_Policy, c(.025))), round(quantile(Time_Cost_Policy, c(.025))),  round(quantile(Prod_Cost_Policy, c(.025)))
                                                          ), nrow=2, ncol = 11, byrow=T)

CE.results_subgroup_UpperBound[,,id_subgroup] <- matrix(c(quantile(LY_disc_No_Policy, c(.975)), quantile(QALY_disc_No_Policy, c(.975)), 
                                                          quantile(CRC_case_No_Policy, c(.975)), quantile(CRC_PY_No_Policy, c(.975)), quantile(CRC_death_No_Policy, c(.975)), 
                                                          quantile(SC_case_No_Policy, c(.975)), quantile(SC_PY_No_Policy, c(.975)), quantile(SC_death_No_Policy, c(.975)), 
                                                          round(quantile(Cost_No_Policy, c(.975))),  round(quantile(Time_Cost_No_Policy, c(.975))), round(quantile(Prod_Cost_No_Policy, c(.975))), 
                                                          quantile(LY_disc_Policy, c(.975)), quantile(QALY_disc_Policy, c(.975)), 
                                                          quantile(CRC_case_Policy, c(.975)), quantile(CRC_PY_Policy, c(.975)), quantile(CRC_death_Policy, c(.975)), 
                                                          quantile(SC_case_Policy, c(.975)), quantile(SC_PY_Policy, c(.975)), quantile(SC_death_Policy, c(.975)), 
                                                          round(quantile(Cost_Policy, c(.975))), round(quantile(Time_Cost_Policy, c(.975))),  round(quantile(Prod_Cost_Policy, c(.975)))
                                                          ), nrow=2, ncol = 11, byrow=T)

CE.incr.results_subgroup[1,,id_subgroup] <- matrix(c(quantile(Incr_LY_disc, c(.025)), quantile(Incr_QALY_disc, c(.025)), 
                                                     quantile(Incr_CRC_case, c(.025)), quantile(Incr_CRC_PY, c(.025)), quantile(Incr_CRC_death, c(.025)), 
                                                     quantile(Incr_SC_case, c(.025)), quantile(Incr_SC_PY, c(.025)), quantile(Incr_SC_death, c(.025)), 
                                                     round(quantile(Incr_Cost, c(.025))), round(quantile(Incr_Time_Cost, c(.025))), round(quantile(Incr_Prod_Cost, c(.025)))
                                                     ))

CE.incr.results_subgroup[2,,id_subgroup] <- matrix(c(mean(Incr_LY_disc), mean(Incr_QALY_disc), 
                                                     mean(Incr_CRC_case), mean(Incr_CRC_PY), mean(Incr_CRC_death), 
                                                     mean(Incr_SC_case), mean(Incr_SC_PY), mean(Incr_SC_death), 
                                                     round(mean(Incr_Cost)), mean(Incr_Time_Cost), round(mean(Incr_Prod_Cost))
                                                     ))

CE.incr.results_subgroup[3,,id_subgroup] <- matrix(c(quantile(Incr_LY_disc, c(.975)), quantile(Incr_QALY_disc, c(.975)), 
                                                     quantile(Incr_CRC_case, c(.975)), quantile(Incr_CRC_PY, c(.975)), quantile(Incr_CRC_death, c(.975)), 
                                                     quantile(Incr_SC_case, c(.975)), quantile(Incr_SC_PY, c(.975)), quantile(Incr_SC_death, c(.975)), 
                                                     round(quantile(Incr_Cost, c(.975))), round(quantile(Incr_Time_Cost, c(.975))), round(quantile(Incr_Prod_Cost, c(.975)))
                                                     ))

}

for (i in 1:32) {
  CE.results_weighted_subgroup[,,i] <- CE.results_subgroup[,,i]*pop_dist[i,2]
  CE.results_weighted_subgroup_LowerBound[,,i] <- CE.results_subgroup_LowerBound[,,i]*pop_dist[i,2]
  CE.results_weighted_subgroup_UpperBound[,,i] <- CE.results_subgroup_UpperBound[,,i]*pop_dist[i,2]
  
  CE.incr.results_weighted_subgroup[,,i] <- CE.incr.results_subgroup[,,i]*pop_dist[i,2]
}

for (r in 1:2){
  for (c in 1:11) {
    CE.results_population[r,c] <- sum(CE.results_weighted_subgroup[r,c,1:32])
    CE.results_population_LowerBound[r,c] <- sum(CE.results_weighted_subgroup_LowerBound[r,c,1:32])
    CE.results_population_UpperBound[r,c] <- sum(CE.results_weighted_subgroup_UpperBound[r,c,1:32])
  }  
}

for (r in 1:3){
  for (c in 1:11) {
    CE.incr.results_population[r,c] <- sum(CE.incr.results_weighted_subgroup[r,c,1:32])
  }  
}

CE.incr.results_population


View(CE.results_population)
View(CE.incr.results_population)
View(CE.results_population_LowerBound)
View(CE.results_population_UpperBound)
